#version 130
#extension GL_EXT_gpu_shader4 : enable
// the version and open GL extension
// should be the first line of the shader
/////////////////////////////////////////////////////////////////////////////////
//Rippled VortexMod01.fsh   by   decrooks
//https://www.shadertoy.com/view/ldjcDK
//Licence : Creative Commons Attribution-ShareAlike 4.0
//http://creativecommons.org/licences/by-sa/4.0
// Adapted, trivialy, for use in VGHD player
/////////////////////////////////////////////
uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

#define iTime u_Elapsed*0.314159  //*0.1666
#define iResolution u_WindowSize

//#define mouse AUTO_MOUSE
//#define MOUSE_SPEED vec2(vec2(0.5,0.577777) * 0.25)
//#define MOUSE_POS   vec2((1.0+cos(iTime*MOUSE_SPEED))*u_WindowSize/2.0)
//#define MOUSE_PRESS vec2(0.0,0.0)
//#define AUTO_MOUSE  vec4( MOUSE_POS, MOUSE_PRESS )
//#define RIGID_SCROLL
// alternatively use static mouse definition
#define iMouse vec4(0.0,0.0, 0.0,0.0)
//#define iMouse vec4(512,256,180,120)
uniform sampler2D iChannel0;
uniform sampler2D iChannel1;
uniform sampler2D iChannel2;
uniform sampler2D iChannel3;
vec4 texture2D_Fract(sampler2D sampler,vec2 P) {return texture2D(sampler,fract(P));}
vec4 texture2D_Fract(sampler2D sampler,vec2 P, float Bias) {return texture2D(sampler,fract(P),Bias);}
#define texture2D texture2D_Fract

// Created by David Crooks
// License Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.

#define TWO_PI 6.283185
#define PI 3.14159265359


struct Ray {
   vec3 origin;
   vec3 direction;
};

struct LightColor {
	vec3 diffuse;
	vec3 specular;
};
    
    
struct Material {
    LightColor  color;
    float shininess;
};
    
struct MapValue {
    float 	  signedDistance;
    Material  material;
};

struct Trace {
    float    dist;
    vec3     p;
    Ray 	 ray;
    Material material;
};
    

struct PointLight {
    vec3 position;
    LightColor color;
};
    
struct DirectionalLight {
    vec3 direction;
    LightColor color;
};
    
PointLight  light1,light2,light3;
DirectionalLight dirLight;

Material blackMat,whiteMat,bluishMat,yellowMat,oscMat,tableMat,tableDarkMat;


    
vec3 rayPoint(Ray r,float t) {
 	return r.origin +  t*r.direction;
}

float bipolarPattern(vec2 p){
    
    float time = iTime;
    
    float scale = ( 40.0+20.0*sin(0.3*time));
    
    //rotate
    float rotationrate = 0.1;
    float s = sin(rotationrate*time);
    float c = cos(rotationrate*time);
    p = mat2(c,s,-s,c)*p;
       
    //bipolar coordinates (sigma, tau)
    float a = min(10.0,2.0*time);
    float alpha = a*a - dot(p,p);
    float beta = a*a + dot(p,p);
    float sigma = atan( 2.0*a*p.y ,alpha);
    float tau = 0.5*log((beta + 2.0*a*p.x)/(beta - 2.0*a*p.x));

    //do something funky in bipolar coordinates
    float freq = 20.0;
    float rate = 3.0;
    vec2 osc = 0.5*(1.0 + cos(freq*vec2(sigma,tau) + rate*time)) ;
    
    //the factor of cosh suppreses the oscilations near the poles where they would otherwise go a bit crazy.
    float cosh = exp(-tau) + exp(tau);
    float bipolarOscillations  = (osc.x + osc.y)/cosh;
    
    return bipolarOscillations;
}

   	

float smoothmin(float a, float b, float k)
{
    float x = exp(-k * a);
    float y = exp(-k * b);
    return (a * x + b * y) / (x + y);
}

float smoothmin2( float a, float b, float k )
{
    float h = clamp( 0.5+0.5*(b-a)/k, 0.0, 1.0 );
    return mix( b, a, h ) - k*h*(1.0-h);
}

float smoothmax(float a, float b, float k)
{
    return smoothmin( a,  b, -k);
}


MapValue intersectObjects( MapValue d1, MapValue d2 )
{
   
    float sd = smoothmax(d1.signedDistance,d2.signedDistance,3.0);
    MapValue mv;
  	mv.material = d1.material;
  	mv.signedDistance = sd;
    return mv;  
    
}



MapValue subtractObjects( MapValue d1, MapValue d2 )
{
    
    d1.signedDistance = -d1.signedDistance;
    return intersectObjects(  d1, d2 );
}


void setMaterials() {
    float t  = iTime;
    float s = 0.4*(1.0+sin(t));
    vec3 specular = vec3(0.3); 
    float shininess = 16.0;
    blackMat = Material(LightColor(vec3(0.0,0.0,0.01),vec3(0.1,0.1,0.1)) ,35.0);
    whiteMat = Material(LightColor(0.75*vec3(1.0,1.0,0.9),0.3*vec3(1.0,1.0,0.9)) ,shininess );

}

/////////////////////////   SDFs   ///////////////////////////////////

MapValue cube( vec3 p, float d , Material m)
{
  MapValue mv;
  mv.material = m;
  mv.signedDistance = length(max(abs(p) -d,0.0));
  return mv; 
}

MapValue xzPlane( vec3 p ,float y, Material m)
{
  MapValue mv;
  mv.material = m;
  mv.signedDistance = p.y - y;
  return mv;
}

MapValue plane(vec3 p, vec3 origin, vec3 normal , Material m ){
  vec3 a = p - origin;
  MapValue mv;
  mv.material = m;
  mv.signedDistance = dot(a,normal);
  return mv;
}

MapValue vortex( vec3 p ,vec3 c, Material m)
{
  MapValue mv;
  mv.material = m;
  mv.signedDistance = length( p.xz - c.xz )  - exp(p.y + c.y);
  return mv;
}


MapValue sphere(vec3 p, vec3 center, float radius, Material m) {
  MapValue mv;
  mv.material = m;
  mv.signedDistance = distance(p, center) - radius;
  return mv;
}

MapValue addObjects(MapValue d1, MapValue d2 )
{
    if (d1.signedDistance<d2.signedDistance) {
    	return    d1 ;
    }
    else {
    	return d2;
    }
}

vec2 orbit( float r, float t){
    return r*vec2(sin(t),cos(t));
}


//////////////////////////////////////////////////////////////////////
/////////////////////// Map The Scene ////////////////////////////////

MapValue map(vec3 p){
    
   	float t  = iTime;
    
    float h = 0.13*bipolarPattern(10.0*p.xz);
    
	p = vec3(p.x,p.y + h,p.z);
    
    vec2 o = orbit(0.5,1.34*iTime);
   
    MapValue vtx  = vortex(p  ,vec3(o.x,-0.2 +0.2*sin(0.45*iTime),o.y), whiteMat);
    
    MapValue pl = plane(p,vec3(0.0,-0.0,0.0),vec3(0.0,1.0,0.0) ,whiteMat);
        
	return subtractObjects( vtx,pl);
    
}


//////////////////////////////////////////////////////////////////////
/////////////////////// Raymarching ///////////////////////////////////

vec3 calculateNormal(vec3 p) {
    float epsilon = 0.001;
    
    vec3 normal = vec3(
                       map(p +vec3(epsilon,0,0)).signedDistance - map(p - vec3(epsilon,0,0)).signedDistance,
                       map(p +vec3(0,epsilon,0)).signedDistance - map(p - vec3(0,epsilon,0)).signedDistance,
                       map(p +vec3(0,0,epsilon)).signedDistance - map(p - vec3(0,0,epsilon)).signedDistance
                       );
    
    return normalize(normal);
}

Trace traceRay(in Ray ray, float maxDistance) {
    float dist = 0.01;
    float presicion = 0.002;
	vec3 p;
    MapValue mv;
    
    for(int i=0; i<64; i++){
    	p = rayPoint(ray,dist);
       	mv = map(p);
         dist += 0.5*mv.signedDistance;
        if(abs(mv.signedDistance) < presicion || dist>maxDistance) break;
       
    }
    
    return Trace(dist,p,ray,mv.material);
}

float castShadow(in Ray ray, float dist){
    Trace trace = traceRay(ray,dist);
    float maxDist = min(1.0,dist);
    float result = trace.dist/maxDist;
   
    return clamp(result,0.0,1.0);
}

Ray cameraRay(vec3 viewPoint, vec3 lookAtCenter, vec2 p , float d){ 
	vec3 v = normalize(lookAtCenter -viewPoint);
    
    vec3 n1 = cross(v,vec3(0.0,1.0,0.0));
    vec3 n2 = cross(n1,v);  
        
    vec3 lookAtPoint = lookAtCenter + d*(p.y*n2 + p.x*n1);
                                    
    Ray ray;
                    
    ray.origin = viewPoint;
   	ray.direction =  normalize(lookAtPoint - viewPoint);
    
    return ray;
}

/////////////////////// Lighting ////////////////////////////////

vec3 diffuseLighting(in Trace trace, vec3 normal, vec3 lightColor,vec3 lightDir){
    float lambertian = max(dot(lightDir,normal), 0.0);
  	return  lambertian * trace.material.color.diffuse * lightColor; 
}

vec3 specularLighting(in Trace trace, vec3 normal, vec3 lightColor,vec3 lightDir){
    //blinn-phong
    //https://en.wikipedia.org/wiki/Blinn%E2%80%93Phong_shading_model
    vec3 viewDir = -trace.ray.direction;

    vec3 halfDir = normalize(lightDir + viewDir);
    float specAngle = max(dot(halfDir, normal), 0.0);
    float specular = pow(specAngle, trace.material.shininess);
    
    return specular * trace.material.color.specular * lightColor;
}


vec3 pointLighting(in Trace trace, vec3 normal, PointLight light){
    vec3 lightDir = light.position - trace.p;
	float d = length(lightDir);
  	lightDir = normalize(lightDir);
   
  	vec3 color =  diffuseLighting(trace, normal, light.color.diffuse, lightDir);
    
    color += specularLighting(trace, normal, light.color.specular, lightDir);

    float  attenuation = 1.0 / (1.0 +  0.1 * d * d);
    float shadow = castShadow(Ray(trace.p,lightDir),d);
    color *= attenuation*shadow;
    return  color;
}

vec3 directionalLighting(Trace trace, vec3 normal, DirectionalLight light){

    vec3 color =  diffuseLighting(trace, normal, light.color.diffuse, light.direction);
    
    color += specularLighting(trace, normal, light.color.specular, light.direction);
    
    float shadow = castShadow(Ray(trace.p,light.direction),3.0);
    color *= shadow;
    return  color;
}


void setLights(){
  	float  time = iTime;
    vec3 specular = vec3(1.0);
    vec3 yellowish = vec3(1.0,1.0,0.7);
    vec3 blueish = vec3(0.7,0.7,1.0);
    
  	light1 = PointLight(vec3(cos(1.3*time),1.0,sin(1.3*time)),LightColor( yellowish,specular));
  	light2 = PointLight(vec3(0.7*cos(1.6*time),1.1+ 0.35*sin(0.8*time),0.7*sin(1.6*time)),LightColor(0.6*yellowish,specular)); 
    light3 = PointLight(vec3(1.5*cos(1.6*time),0.15+ 0.15*sin(2.9*time),1.5*sin(1.6*time)),LightColor(0.5*blueish,specular));
    dirLight = DirectionalLight(normalize(vec3(0.0,1.0,0.0)),LightColor(vec3(0.1),vec3(0.5)));
} 


vec3 lighting(in Trace trace, vec3 normal){
    vec3 color = vec3(0.0,0.0,0.2);//ambient color, dark blue
        
	color += pointLighting(trace, normal,light1);
	color += pointLighting(trace, normal,light2) ;
    color += pointLighting(trace, normal,light3) ;
	color += directionalLighting(trace, normal,dirLight);
    
    return color;
}

vec3 render(vec2 p){
    vec3 viewpoint = vec3(-1.0,1.3,-1.3);
    
    vec3 lookAt = vec3(0.0,-0.1,0.0);
    
  	Ray ray = cameraRay(viewpoint,lookAt,p,2.3);
   	Trace trace = traceRay(ray,6.0);
    
	vec3 normal = calculateNormal(trace.p);
    vec3 color = lighting(trace,normal);
    
   	return color;
}

//void mainImage( out vec4 fragColor, in vec2 fragCoord )
///////////////////////////////////////////////////////////////////////////////// 
// need to convert this from a void to a function and call it by adding
// a void main(void) { to the end of the shader
// what type of variable will the function return?, it is a color and needs to be a vec4
// change void to vec4 
//void MainImage(out vec4 fragColor, in vec2 fragCoord) 
vec4 mainImage( out vec4 fragColor, in vec2 fragCoord )
{ 
	vec2 uv = fragCoord.xy / iResolution.xy;
    vec2 p = (fragCoord - 0.5*iResolution.xy) / iResolution.y;
    
  	setLights();
    setMaterials();
    
   	vec3 colorLinear =  render(p);
    float screenGamma = 2.2;
    vec3 colorGammaCorrected = pow(colorLinear, vec3(1.0/screenGamma));
	fragColor = vec4(colorGammaCorrected,1.0);
/////////////////////////////////////////////////////////////////////////////////
//the function needs to return a value. 
//it needs to be a vec4
//we will return the varable fragColor 
// usual place for fragColor = vec4( color, 1.0 ); bring the } down below
return fragColor; 
}

///////////////////////////////////////////////////////////////////////////////// 
void main(void) { // this will be run for every pixel of gl_FragCoord.xy
vec4 fragColor = vec4(1.0); // initialize variable fragColor as a vec4 
vec4 cc = mainImage(fragColor, gl_FragCoord.xy); // call function mainImage and assign the return vec4 to cc
gl_FragColor = vec4(cc) * gl_Color; // set the pixel to the value of vec4 cc  and..
}

// ..uses the values of any Color: or Opacity:
// clauses (and any Animate clauses applied to these properties) 
// appearing in the Sprite, Quad or other node invoking the shader 
// in the .scn file.

